<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Settings;
use App\Models\Shipment;

class ShipmentController extends Controller
{
    // List all shipments
    public function index()
    {
        $shipments = Shipment::orderByDesc('id')->paginate(20);
        return view('admin.shipments.index', compact('shipments'));
    }

    // Show create shipment form
    public function create()
    {
        $settings = Settings::where('id',1)->first();
        $siteInitials = strtoupper(substr(preg_replace('/\s+/', '', $settings->site_name), 0, 3));
        // Generate a unique tracking number (e.g., SIT123456)
        do {
            $trackingnumber = $siteInitials . random_int(100000, 999999);
        } while (\App\Models\Shipment::where('trackingnumber', $trackingnumber)->exists());
        $title = 'Create New Shipment';
        return view('admin.createshipment', compact('trackingnumber', 'settings', 'title'));
    }

    // Store new shipment
    public function store(Request $request)
    {
        $validated = $request->validate([
            'trackingnumber' => 'required|unique:shipments|max:100',
            'sender_name' => 'required',
            'sender_address' => 'required',
            'sender_email' => 'required|email',
            'sender_phone' => 'required',
            'receiver_name' => 'required',
            'receiver_address' => 'required',
            'receiver_email' => 'required|email',
            'receiver_phone' => 'required',
            'origin' => 'required',
            'destination' => 'required',
            'shipment_type' => 'required',
            'freight_type' => 'required',
            'weight' => 'required',
            'date_shipped' => 'required|date',
            'expected_delivery' => 'required|date',
            'status' => 'required',
            'description' => 'nullable',
            'photo_file' => 'nullable|file|image|max:2048',
            'location' => 'nullable|string',
            'addresslocation' => 'required|string',
            'cost' => 'nullable|numeric',
            'clearance_cost' => 'nullable|numeric',
            'qty' => 'nullable|integer',
            'percentage_complete' => 'nullable|integer',
        ]);

        // Handle file upload
        if ($request->hasFile('photo_file')) {
            $photoPath = $request->file('photo_file')->store('photos', 'public');
            $validated['photo'] = $photoPath;
        }

        // Remove photo_file from validated data since it's not a database field
        unset($validated['photo_file']);

        Shipment::create($validated);
        return redirect()->route('admin.shipments.index')->with('success', 'Shipment created successfully.');
    }

    // Show edit shipment form
    public function edit($id)
    {
        $shipment = Shipment::findOrFail($id);
        return view('admin.editshipment', compact('shipment'));
    }

    // Update shipment
    public function update(Request $request, $id)
    {
        $shipment = Shipment::findOrFail($id);
        $validated = $request->validate([
            'trackingnumber' => 'required|max:100|unique:shipments,trackingnumber,' . $shipment->id,
            'sender_name' => 'required',
            'sender_address' => 'required',
            'sender_email' => 'required|email',
            'sender_phone' => 'required',
            'receiver_name' => 'required',
            'receiver_address' => 'required',
            'receiver_email' => 'required|email',
            'receiver_phone' => 'required',
            'origin' => 'required',
            'destination' => 'required',
            'shipment_type' => 'required',
            'freight_type' => 'required',
            'weight' => 'required',
            'date_shipped' => 'required|date',
            'expected_delivery' => 'required|date',
            'status' => 'required',
            'description' => 'nullable',
            'photo' => 'nullable|string',
            'location' => 'nullable|string',
            'percentage_complete' => 'nullable|integer',
        ]);
        $shipment->update($validated);
        return redirect()->route('admin.shipments.index')->with('success', 'Shipment updated successfully.');
    }

    // View a single shipment
    public function show($id)
    {
        $shipment = Shipment::findOrFail($id);
        return view('admin.shipments.show', compact('shipment'));
    }

    // Show update shipment status form
    public function showUpdateForm($id)
    {
        $shipment = Shipment::findOrFail($id);
        return view('admin.updateshipment', compact('shipment'));
    }

    // Update only shipment status, location, progress, and description
    public function updateStatus(Request $request, $id)
    {
        $shipment = Shipment::findOrFail($id);
        $validated = $request->validate([
            'status' => 'required',
            'location' => 'nullable|string',
            'percentage_complete' => 'nullable|integer|min:0|max:100',
            'description' => 'nullable|string',
        ]);
        $shipment->update($validated);
        return redirect()->route('admin.shipments.show', $shipment->id)->with('success', 'Shipment status updated successfully.');
    }
}
